/*
   *  Object %name    : KMNG.c
   *  State           :  %state%
   *  Creation date   :  Wed Nov 17 17:39:24 2004
   *  Last modified   :  %modify_time%
   */
  /** @file
   *  \brief Secure Boot implemtation of hash queues
   *
   *  \version 
   *  \author yevgenys
   *  \remarks Copyright (C) 2004 by Discretix Technologies Ltd.
   *           All Rights reserved
   */


#include "DX_VOS_BaseTypes.h"
#include "DX_VOS_Mem.h"
#include "DX_VOS_Memmap.h"
#include "CRYS.h"
#include "KMNG_Defs.h"
#include "LLF_KMNG_DEF.h"
#include "LLF_KMNG_ERR.h"
#include "error.h"
#include "CRYS_RSA_Types.h"
#include "CRYS_AES.h"
#include "KMNG_API.h"
#ifdef KMNG_SEP_SIMULATOR
#include "llf_kmng_wrapper.h"
#endif
#include "LLF_KMNG.h"
#include "host_op_code.h"
#include "kmng_host_op_code.h"
#include "SEPDriver.h"


/*-------------------------
  PROTOTYPES
--------------------------*/

static DxError_t LLF_KMNG_SendCreateUserKeyMessage(KMNG_KeyDataParameter_t*   keyRingData_ptr,
                                                   KMNG_KeyDataParameter_t*   newKeyData_ptr,
                                                   KMNG_KeyConfiguration_t*   keyConfiguration_ptr);
                                           
                                           
static DxError_t LLF_KMNG_SendCreateKeyRingMessage(KMNG_KeyDataParameter_t*   newKeyData_ptr,
                                                   KMNG_KeyConfiguration_t*   keyConfiguration_ptr);
   

static void LLF_KMNG_SendKeyRingData(KMNG_KeyDataParameter_t*    keyRingData_ptr,
                                          DxUint32_t*                 sramOffset_ptr);

/*---------------------
    FUNCTIONS
-----------------------*/

/**
 * @brief     This function initializes the LLF KMNG module
 *
 * @return     DxError:  
 *                        
 */
DxError_t LLF_KMNG_Init()
{
  /* sram offset */
  DxUint32_t  sramOffset;
  
  /* message param */
  DxUint32_t  messageParam[2];
  
  /* error */
  DxError_t   error;

  /*-------------------------------
      CODE
  ----------------------------------*/
  
  error = KMNG_OK;
  
  /* lock access to the SEP */
  error = SEPDriver_Lock();
   
  if(error != DX_OK)
  {
    goto end_function;
  }
  
  /*----------------------------
    start sending message to SEP 
  -----------------------------*/

  sramOffset = 0;
   
  /* start the message */
  SEPDriver_StartMessage(&sramOffset);
  
  /* prepare message  */
  messageParam[0] = DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_INIT_OP_CODE;
 
  /* send message */
  error = SEPDriver_WriteParamater((DxUint32_t)messageParam,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
             
  SEPDriver_EndMessage(sramOffset);
            
  /* wait for the response */
  error = SEPDriver_POLL_FOR_REPONSE();
  if( error != DX_OK )
  {
    goto end_function_unlock;
  }
   
  /*-------------------
    start reading message from the SEP 
  ---------------------*/
   
  /* start the message */
  error = SEPDriver_StartIncomingMessage(&sramOffset);
  if( error != DX_OK )
  {
    goto end_function_unlock;
  }
   
  /* read opcode + status */
  error = SEPDriver_ReadParamater((DxUint32_t)messageParam,
                          sizeof(DxUint32_t) * 2,
                          sizeof(DxUint32_t) * 2, 
                          &sramOffset , 
                          DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
   
  /* check the opcode */
  if(messageParam[0] != DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_INIT_OP_CODE)
  {
    error = DX_WRONG_OPCODE_FROM_SEP_ERR;
    
  }
  
  error = messageParam[1];

end_function_unlock:   

  /* unlock access to the SEP */
  SEPDriver_Unlock();
  
end_function:
  
  return error;
}

DxError_t LLF_KMNG_CreateKey(KMNG_KeyDataParameter_t *keyRingKeyData_ptr,
                             KMNG_KeyDataParameter_t *newKeyData_ptr,
                             KMNG_KeyConfiguration_t *keyConfiguration_ptr,
							               KMNG_BufferData_t		 *KeyGenData_ptr )
{
  /* error */
  DxError_t       error;
  
  /*----------------------------
      CODE
  -----------------------------*/
  
  KeyGenData_ptr = KeyGenData_ptr;
  
  /* initializing the Error to O.K */
  error = KMNG_OK;

                                            
      
  /* ............... checking the parameters validity ................... */
  /* -------------------------------------------------------------------- */
   
  /* if the users context ID pointer is DX_NULL return an error */
  if( newKeyData_ptr == DX_NULL ||
      keyConfiguration_ptr == DX_NULL )
  {
    error = KMNG_LLF_KEY_STRUCT_NULL_ERR;
    goto end_function;
  }
  
  if(keyRingKeyData_ptr == DX_NULL)
  {
    /* create the protection key of the Key Ring */
    error = LLF_KMNG_SendCreateKeyRingMessage(newKeyData_ptr , keyConfiguration_ptr);
  }
  else
  {
    /* create the user key  */
    error = LLF_KMNG_SendCreateUserKeyMessage(keyRingKeyData_ptr , newKeyData_ptr , keyConfiguration_ptr);
  }

end_function:

  return error;
}
                          

/**
 * @brief     This function imports a symmetric key into key ring
 *
 *
 * @param[in] keyRingKeyData_ptr - pointer to the key data of the Key Ring
 * @param[in] keyConfiguration_ptr - configuration of the imported key
 * @param[in/out] importedKeyData_ptr - imported key data. On input , only password and password length fields are set
 * @param[in] importSymKey_ptr - symmtric key data to be imported
 * 
 * @return     DxError:  
 *                        
 */
DxError_t LLF_KMNG_ImportSymUserKey(KMNG_KeyDataParameter_t*    keyRingKeyData_ptr,
                                    KMNG_KeyConfiguration_t*    keyConfiguration_ptr,
                                    KMNG_KeyDataParameter_t*    importedKeyData_ptr,
                                    DxUint8_t*                  importSymKey_ptr)
{
  /* sram offset */
  DxUint32_t  sramOffset;
  
  /* message param */
  DxUint32_t  messageParam[3];
  
  /* max length */
  DxUint32_t  maxLength;
  
  /* key length */
  DxUint32_t  keyLength;
  
  /* error */
  DxError_t   error;

  /*-------------------------------
      CODE
  ----------------------------------*/
  
  error = KMNG_OK;
  
  
  /* key length in bytes */
  keyLength = keyConfiguration_ptr->keySizeBytes;
  
   /* lock access to the SEP */
   error = SEPDriver_Lock();
   
   if(error != DX_OK)
   {
       goto end_function;
   }
  
  /*----------------------------
      start sending message to SEP 
  -----------------------------*/
  sramOffset = 0;
   
  /* start the message */
  SEPDriver_StartMessage(&sramOffset);
  
  /* prepare message  */
  messageParam[0] = DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_IMPORT_SYM_KEY_OP_CODE;
 
  /* send message */
  error = SEPDriver_WriteParamater((DxUint32_t)messageParam,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
 
  /* send key ring data */                   
  LLF_KMNG_SendKeyRingData(keyRingKeyData_ptr , &sramOffset);
  
  /* key password length */ 
  error = SEPDriver_WriteParamater((DxUint32_t)&importedKeyData_ptr->keyPasswordLen,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
  
  /* send key password */                         
  error = SEPDriver_WriteParamater((DxUint32_t)importedKeyData_ptr->keyPassword_ptr,
                           importedKeyData_ptr->keyPasswordLen,
                           64*sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
  
  /* send key configuration */
  error = SEPDriver_WriteParamater((DxUint32_t)keyConfiguration_ptr,
                           sizeof(DxUint32_t) * 6,
                           sizeof(DxUint32_t) * 6,
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                           
  /* send key length */
  error = SEPDriver_WriteParamater((DxUint32_t)&keyLength,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                           
  /* send key */
  maxLength = ((keyLength + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
  error = SEPDriver_WriteParamater((DxUint32_t)importSymKey_ptr,
                           keyLength,
                           maxLength,
                           &sramOffset , 
                           DX_TRUE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
  
                            
  SEPDriver_EndMessage(sramOffset);
            
  /* wait for the response */
  error = SEPDriver_POLL_FOR_REPONSE();
  if( error != DX_OK )
  {
    goto end_function_unlock;
  }
  
   
  /*-------------------
    start reading message from the SEP 
  ---------------------*/
   
  /* start the message */
  error = SEPDriver_StartIncomingMessage(&sramOffset);
  if( error != DX_OK )
  {
    goto end_function_unlock;
  }
   
  /* read opcode + status */
  error = SEPDriver_ReadParamater((DxUint32_t)messageParam,
                          sizeof(DxUint32_t) * 3,
                          sizeof(DxUint32_t) * 3, 
                          &sramOffset , 
                          DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
   
  /* check the opcode */
  if(messageParam[0] != DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_IMPORT_SYM_KEY_OP_CODE)
  {
    error = DX_WRONG_OPCODE_FROM_SEP_ERR;
    goto end_function_unlock;
  }
   
  /* check the status */
  if(messageParam[1] != CRYS_OK)
  {
    error = messageParam[1];
    goto end_function_unlock;
  }
  
  /* get the length of the encrypted data  */
  importedKeyData_ptr->encryptedKeyDataLength = messageParam[2];
  
  /* read the encrypted key */
  maxLength = ((importedKeyData_ptr->encryptedKeyDataLength + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
  error = SEPDriver_ReadParamater((DxUint32_t)importedKeyData_ptr->encryptedKeyData_ptr,
                          importedKeyData_ptr->encryptedKeyDataLength,
                          maxLength, 
                          &sramOffset , 
                          DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }

end_function_unlock:

  /* unlock access to the SEP */
  SEPDriver_Unlock();

end_function:                           

  return error;
}
                                       
                                                                       
/**
 * @brief     This function imports a RSA key into key ring
 *
 *
 * @param[in] keyRingKeyData_ptr - pointer to the key data of the Key Ring
 * @param[in] keyConfiguration_ptr - configuration of the imported key
 * @param[in/out] importedKeyData_ptr - imported key data. On input , only password and password length fields are set
 * @param[in] modulus_ptr - modulus pointer
 * @param[in] modulusSizeInBytes - modulus size
 * @param[in] E_ptr - E pointer
 * @param[in] E_SizeInBytes - E size
 * @param[in] D_ptr - D pointer
 * @param[in] D_SizeInBytes - D size
 * 
 * @return     DxError:  
 *                        
 */
DxError_t LLF_KMNG_ImportRSAUserKey(KMNG_KeyDataParameter_t*    keyRingKeyData_ptr,
                                    KMNG_KeyConfiguration_t*    keyConfiguration_ptr,
                                    KMNG_KeyDataParameter_t*    importedKeyData_ptr,
                                    DxUint8_t*                  modulus_ptr,
                                    DxUint32_t                  modulusSizeInBytes,
                                    DxUint8_t*                  E_ptr,
                                    DxUint32_t                  E_SizeInBytes,
                                    DxUint8_t*                  D_ptr,
                                    DxUint32_t                  D_SizeInBytes)
{
  /* sram offset */
  DxUint32_t  sramOffset;
  
  /* message param */
  DxUint32_t  messageParam[3];
  
  /* max length */
  DxUint32_t  maxLength;
  
  /* error */
  DxError_t   error;

  /*-------------------------------
      CODE
  ----------------------------------*/
  
  error = KMNG_OK;
  
   /* lock access to the SEP */
   error = SEPDriver_Lock();
   
   if(error != DX_OK)
   {
       goto end_function;
   }
  
  /*----------------------------
      start sending message to SEP 
  -----------------------------*/
  sramOffset = 0;
   
  /* start the message */
  SEPDriver_StartMessage(&sramOffset);
  
  /* prepare message  */
  messageParam[0] = DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_IMPORT_RSA_KEY_OP_CODE;
 
  /* send message */
  error = SEPDriver_WriteParamater((DxUint32_t)messageParam,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
 
  /* send key ring data */                   
  LLF_KMNG_SendKeyRingData(keyRingKeyData_ptr , &sramOffset);

  /* key password length */ 
  error = SEPDriver_WriteParamater((DxUint32_t)&importedKeyData_ptr->keyPasswordLen,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }

  /* send key password */                         
  error = SEPDriver_WriteParamater((DxUint32_t)importedKeyData_ptr->keyPassword_ptr,
                           importedKeyData_ptr->keyPasswordLen,
                           64*sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
  
  /* send configuration */
  error = SEPDriver_WriteParamater((DxUint32_t)keyConfiguration_ptr,
                           sizeof(DxUint32_t) * 6,
                           sizeof(DxUint32_t) * 6,
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                           
  /* send modulus length  */
  error = SEPDriver_WriteParamater((DxUint32_t)&modulusSizeInBytes,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }

  /* send modulus */
  error = SEPDriver_WriteParamater((DxUint32_t)modulus_ptr,
                           modulusSizeInBytes,
                           64*sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_TRUE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                           
  /* send E length  */
  error = SEPDriver_WriteParamater((DxUint32_t)&E_SizeInBytes,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                           
  /* send E */
  error = SEPDriver_WriteParamater((DxUint32_t)E_ptr,
                           E_SizeInBytes,
                           64*sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_TRUE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }

  /* send D length  */
  error = SEPDriver_WriteParamater((DxUint32_t)&D_SizeInBytes,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                           
  /* send D */
  error = SEPDriver_WriteParamater((DxUint32_t)D_ptr,
                           D_SizeInBytes,
                           64*sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_TRUE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                           
  SEPDriver_EndMessage(sramOffset);
            
  /* wait for the response */
  error = SEPDriver_POLL_FOR_REPONSE();
  if( error != DX_OK )
  {
    goto end_function_unlock;
  }
   
  /*-------------------
    start reading message from the SEP 
  ---------------------*/
   
  /* start the message */
  error = SEPDriver_StartIncomingMessage(&sramOffset);
  if( error != DX_OK )
  {
    goto end_function_unlock;
  }
   
  /* read opcode + status */
  error = SEPDriver_ReadParamater((DxUint32_t)messageParam,
                          sizeof(DxUint32_t) * 3,
                          sizeof(DxUint32_t) * 3, 
                          &sramOffset , 
                          DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
   
  /* check the opcode */
  if(messageParam[0] != DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_IMPORT_RSA_KEY_OP_CODE)
  {
    error = DX_WRONG_OPCODE_FROM_SEP_ERR;
    goto end_function_unlock;
  }
   
  /* check the status */
  if(messageParam[1] != CRYS_OK)
  {
    error = messageParam[1];
    goto end_function_unlock;
  }
  
  /* get the length of the encrypted data  */
  importedKeyData_ptr->encryptedKeyDataLength = messageParam[2];
  
  /* read the encrypted key */
  maxLength = ((importedKeyData_ptr->encryptedKeyDataLength + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
  error = SEPDriver_ReadParamater((DxUint32_t)importedKeyData_ptr->encryptedKeyData_ptr,
                          importedKeyData_ptr->encryptedKeyDataLength,
                          maxLength, 
                          &sramOffset , 
                          DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }

end_function_unlock:

  /* unlock access to the SEP */
  SEPDriver_Unlock();

end_function:

  return error;
}
                                       

/**
 * @brief     This function imports a DH key into key ring
 *
 *
 * @param[in] keyRingKeyData_ptr - pointer to the key data of the Key Ring
 * @param[in] keyConfiguration_ptr - configuration of the imported key
 * @param[in/out] importedKeyData_ptr - imported key data. On input , only password and password length fields are set
 * @param[in] modulus_ptr - modulus pointer
 * @param[in] modulusSize - modulus size
 * @param[in] generator_ptr - generator pointer
 * @param[in] generatorSize - generator size
 * @param[in] E_ptr - E pointer
 * @param[in] E_Size - E size
 * @param[in] D_ptr - D pointer
 * @param[in] D_Size - D size
 * 
 * @return     DxError:  
 *                        
 */                                       
DxError_t LLF_KMNG_ImportDHUserKey(KMNG_KeyDataParameter_t*   keyRingKeyData_ptr,
                                   KMNG_KeyConfiguration_t*   keyConfiguration_ptr,
                                   KMNG_KeyDataParameter_t*   importedKeyData_ptr,
                                   DxUint8_t*                 modulus_ptr,
                                   DxUint32_t                 modulusSize,
                                   DxUint8_t*                 generator_ptr,
                                   DxUint32_t                 generatorSize,
                                   DxUint8_t*                 E_ptr,
                                   DxUint32_t                 E_Size,
                                   DxUint8_t*                 D_ptr,
                                   DxUint32_t                 D_Size)
{
  /* sram offset */
  DxUint32_t  sramOffset;
  
  /* message param */
  DxUint32_t  messageParam[3];
  
  /* max length */
  DxUint32_t  maxLength;
  
  /* error */
  DxError_t   error;

  /*-------------------------------
      CODE
  ----------------------------------*/
  
  error = KMNG_OK;
  
   /* lock access to the SEP */
   error = SEPDriver_Lock();
   
   if(error != DX_OK)
   {
       goto end_function;
   }
  
  /*----------------------------
      start sending message to SEP 
  -----------------------------*/
  sramOffset = 0;
   
  /* start the message */
  SEPDriver_StartMessage(&sramOffset);
  
  /* prepare message  */
  messageParam[0] = DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_IMPORT_DH_KEY_OP_CODE;
 
  /* send message */
  error = SEPDriver_WriteParamater((DxUint32_t)messageParam,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
 
  /* send key ring data */                   
  LLF_KMNG_SendKeyRingData(keyRingKeyData_ptr , &sramOffset);

  /* key password length */ 
  error = SEPDriver_WriteParamater((DxUint32_t)&importedKeyData_ptr->keyPasswordLen,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
  
  /* send key password */                         
  error = SEPDriver_WriteParamater((DxUint32_t)importedKeyData_ptr->keyPassword_ptr,
                           importedKeyData_ptr->keyPasswordLen,
                           64*sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }

  /* send configuration */
  error = SEPDriver_WriteParamater((DxUint32_t)keyConfiguration_ptr,
                           sizeof(DxUint32_t) * 6,
                           sizeof(DxUint32_t) * 6,
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                           
  /* send modulus length  */
  error = SEPDriver_WriteParamater((DxUint32_t)&modulusSize,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                           
  /* send modulus */
  error = SEPDriver_WriteParamater((DxUint32_t)modulus_ptr,
                           modulusSize,
                           64*sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_TRUE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                           
  /* send generator length  */
  error = SEPDriver_WriteParamater((DxUint32_t)&generatorSize,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                           
  /* send generator */
  error = SEPDriver_WriteParamater((DxUint32_t)generator_ptr,
                           generatorSize,
                           64*sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_TRUE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                           
  /* send E length  */
  error = SEPDriver_WriteParamater((DxUint32_t)&E_Size,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                           
  /* send E */
  error = SEPDriver_WriteParamater((DxUint32_t)E_ptr,
                           E_Size,
                           64*sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_TRUE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }

  /* send D length  */
  error = SEPDriver_WriteParamater((DxUint32_t)&D_Size,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                           
  /* send D */
  error = SEPDriver_WriteParamater((DxUint32_t)D_ptr,
                           D_Size,
                           64*sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_TRUE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                           
  SEPDriver_EndMessage(sramOffset);
            
  /* wait for the response */
  error = SEPDriver_POLL_FOR_REPONSE();
  if( error != DX_OK )
  {
    goto end_function_unlock;
  }
   
  /*-------------------
    start reading message from the SEP 
  ---------------------*/
   
  /* start the message */
  error = SEPDriver_StartIncomingMessage(&sramOffset);
  if( error != DX_OK )
  {
    goto end_function_unlock;
  }
   
  /* read opcode + status */
  error = SEPDriver_ReadParamater((DxUint32_t)messageParam,
                          sizeof(DxUint32_t) * 3,
                          sizeof(DxUint32_t) * 3, 
                          &sramOffset , 
                          DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
   
  /* check the opcode */
  if(messageParam[0] != DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_IMPORT_DH_KEY_OP_CODE)
  {
    error = DX_WRONG_OPCODE_FROM_SEP_ERR;
    goto end_function_unlock;
  }
   
  /* check the status */
  if(messageParam[1] != CRYS_OK)
  {
    error = messageParam[1];
    goto end_function_unlock;
  }
  
  /* get the length of the encrypted data  */
  importedKeyData_ptr->encryptedKeyDataLength = messageParam[2];
  
  /* read the encrypted key */
  maxLength = ((importedKeyData_ptr->encryptedKeyDataLength + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
  error = SEPDriver_ReadParamater((DxUint32_t)importedKeyData_ptr->encryptedKeyData_ptr,
                          importedKeyData_ptr->encryptedKeyDataLength,
                          maxLength, 
                          &sramOffset , 
                          DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }

end_function_unlock:

  /* unlock access to the SEP */
  SEPDriver_Unlock();

end_function:
                           
  return error;
}


/**
 * @brief     This function exports a symmetric key from key ring
 *
 *
 * @param[in] keyRingKeyData_ptr - pointer to the key data of the Key Ring
 * @param[out] keyConfiguration_ptr - configuration of the exported key
 * @param[in] exportKeyData_ptr - ipointer to the data of the key to be exported
 * @param[out] keyData_ptr - symmetric key itself after decryption
 * 
 * @return     DxError:  
 *                        
 */                                       
DxError_t  _LLF_KMNG_ExportSymUserKey(KMNG_KeyDataParameter_t*   keyRingKeyData_ptr,
                                      KMNG_KeyConfiguration_t*   keyConfiguration_ptr,
                                      KMNG_KeyDataParameter_t*   exportKeyData_ptr,
                                      DxUint8_t*                 keyData_ptr,
									                    DxUint32_t*                keyDataLen_ptr,
									                    DxUint32_t                 exportInternalFlag)
{
  /* sram offset */
  DxUint32_t  sramOffset;
  
  /* message param */
  DxUint32_t  messageParam[9];
  
  /* max length */
  DxUint32_t  maxLength;
  
  /* error */
  DxError_t   error;

  /*-------------------------------
      CODE
  ----------------------------------*/

  keyConfiguration_ptr = keyConfiguration_ptr;
  keyDataLen_ptr = keyDataLen_ptr;
  
  error = KMNG_OK;
  
   /* lock access to the SEP */
   error = SEPDriver_Lock();
   
   if(error != DX_OK)
   {
       goto end_function;
   }
  
  /*----------------------------
      start sending message to SEP 
  -----------------------------*/
  sramOffset = 0;
   
  /* start the message */
  SEPDriver_StartMessage(&sramOffset);
  
  /* prepare message  */
  messageParam[0] = DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_EXPORT_SYM_KEY_OP_CODE;
 
  /* send message */
  error = SEPDriver_WriteParamater((DxUint32_t)messageParam,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
 
  /* send key ring data */                   
  LLF_KMNG_SendKeyRingData(keyRingKeyData_ptr , &sramOffset);
  
  /* key password length */ 
  error = SEPDriver_WriteParamater((DxUint32_t)&exportKeyData_ptr->keyPasswordLen,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
  
  /* send key password */                         
  error = SEPDriver_WriteParamater((DxUint32_t)exportKeyData_ptr->keyPassword_ptr,
                           exportKeyData_ptr->keyPasswordLen,
                           64*sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                           
  /* send sym key data length */
  error = SEPDriver_WriteParamater((DxUint32_t)&exportKeyData_ptr->encryptedKeyDataLength,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
  
                           
  /* send sym key data */
  maxLength = ((exportKeyData_ptr->encryptedKeyDataLength + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
  error = SEPDriver_WriteParamater((DxUint32_t)exportKeyData_ptr->encryptedKeyData_ptr,
                           exportKeyData_ptr->encryptedKeyDataLength,
                           maxLength,
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                           
  SEPDriver_EndMessage(sramOffset);
            
  /* wait for the response */
  error = SEPDriver_POLL_FOR_REPONSE();
  if( error != DX_OK )
  {
    goto end_function_unlock;
  }
   
  /*-------------------
    start reading message from the SEP 
  ---------------------*/
   
  /* start the message */
  error = SEPDriver_StartIncomingMessage(&sramOffset);
  if( error != DX_OK )
  {
    goto end_function_unlock;
  }
   
  /* read opcode + status */
  error = SEPDriver_ReadParamater((DxUint32_t)messageParam,
                          sizeof(DxUint32_t) * 6,
                          sizeof(DxUint32_t) * 6, 
                          &sramOffset , 
                          DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
   
  /* check the opcode */
  if(messageParam[0] != DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_EXPORT_SYM_KEY_OP_CODE)
  {
    error = DX_WRONG_OPCODE_FROM_SEP_ERR;
    goto end_function_unlock;
  }
   
  /* check the status */
  if(messageParam[1] != KMNG_OK)
  {
    error = messageParam[1];
    goto end_function_unlock;
  }
  
  /* read configuration */
  keyConfiguration_ptr->type = (KMNG_KeyType_t)messageParam[2];
  keyConfiguration_ptr->restriction = messageParam[3];
  keyConfiguration_ptr->usage = messageParam[4];
  keyConfiguration_ptr->keySizeBytes = messageParam[5];

  /* read user specific key data */  
   error = SEPDriver_ReadParamater((DxUint32_t)keyConfiguration_ptr->UserSpecificKeyData,
                          KMNG_USER_SPECIFIC_KEY_DATA_SIZE,
                          KMNG_USER_SPECIFIC_KEY_DATA_SIZE, 
                          &sramOffset , 
                          DX_TRUE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }

   
  /* read the symmetric key key */
  maxLength = ((messageParam[5] + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
  error = SEPDriver_ReadParamater((DxUint32_t)keyData_ptr,
                          messageParam[5],
                          maxLength, 
                          &sramOffset , 
                          DX_TRUE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }

end_function_unlock:

  /* unlock access to the SEP */
  SEPDriver_Unlock();

end_function:

  return error;
}

/**
 * @brief     This function exports an RSA key from key ring
 *
 *
 * @param[in] keyRingKeyData_ptr - pointer to the key data of the Key Ring
 * @param[out] keyConfiguration_ptr - configuration of the exported key
 * @param[in] exportKeyData_ptr - ipointer to the data of the key to be exported
 * @param[out] modulus_ptr - modulus pointer
 * @param[out] modulusSize_ptr - modulus size
 * @param[out] E_ptr - N pointer
 * @param[out] E_Size_ptr - N size
 * @param[out] D_ptr - D pointer
 * @param[out] D_Size_ptr - D size
 * 
 * @return     DxError:  
 *                        
 */                                           
DxError_t _LLF_KMNG_ExportRSAUserKey(KMNG_KeyDataParameter_t*    keyRingKeyData_ptr,
                                    KMNG_KeyConfiguration_t*    keyConfiguration_ptr,
                                    KMNG_KeyDataParameter_t*    exportKeyData_ptr,
                                    DxUint8_t*                  modulus_ptr,
                                    DxUint32_t*                 modulusSize_ptr,
                                    DxUint8_t*                  E_ptr,
                                    DxUint32_t*                 E_Size_ptr,
                                    DxUint8_t*                  D_ptr,
                                    DxUint32_t*                 D_Size_ptr,
                                    DxUint32_t                 exportInternalFlag)
{
  /* sram offset */
  DxUint32_t  sramOffset;
  
  /* message param */
  DxUint32_t  messageParam[3];
  
  /* max length */
  DxUint32_t  maxLength;
  
  /* error */
  DxError_t   error;

  /*-------------------------------
      CODE
  ----------------------------------*/
  
  modulus_ptr = modulus_ptr;
  keyConfiguration_ptr = keyConfiguration_ptr;

  error = KMNG_OK;

   /* lock access to the SEP */
   error = SEPDriver_Lock();
   
   if(error != DX_OK)
   {
       goto end_function;
   }
  
  /*----------------------------
      start sending message to SEP 
  -----------------------------*/
  sramOffset = 0;
   
  /* start the message */
  SEPDriver_StartMessage(&sramOffset);
  
  /* prepare message  */
  messageParam[0] = DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_EXPORT_RSA_KEY_OP_CODE;
 
  /* send message */
  error = SEPDriver_WriteParamater((DxUint32_t)messageParam,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
 
  /* send key ring data */                   
  LLF_KMNG_SendKeyRingData(keyRingKeyData_ptr , &sramOffset);
  
  /* key password length */ 
  error = SEPDriver_WriteParamater((DxUint32_t)&exportKeyData_ptr->keyPasswordLen,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
  
  /* send key password */                         
  error = SEPDriver_WriteParamater((DxUint32_t)exportKeyData_ptr->keyPassword_ptr,
                           exportKeyData_ptr->keyPasswordLen,
                           64*sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                           
  /* send RSA key data length */
  error = SEPDriver_WriteParamater((DxUint32_t)&exportKeyData_ptr->encryptedKeyDataLength,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
  
                           
  /* send RSA key data */
  maxLength = ((exportKeyData_ptr->encryptedKeyDataLength + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
  error = SEPDriver_WriteParamater((DxUint32_t)exportKeyData_ptr->encryptedKeyData_ptr,
                           exportKeyData_ptr->encryptedKeyDataLength,
                           maxLength,
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                           
  SEPDriver_EndMessage(sramOffset);
            
  /* wait for the response */
  error = SEPDriver_POLL_FOR_REPONSE();
  if( error != DX_OK )
  {
    goto end_function_unlock;
  }
   
  /*-------------------
    start reading message from the SEP 
  ---------------------*/
   
  /* start the message */
  error = SEPDriver_StartIncomingMessage(&sramOffset);
  if( error != DX_OK )
  {
    goto end_function_unlock;
  }
   
  /* read opcode + status */
  error = SEPDriver_ReadParamater((DxUint32_t)messageParam,
                          sizeof(DxUint32_t) * 3,
                          sizeof(DxUint32_t) * 3, 
                          &sramOffset , 
                          DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
   
  /* check the opcode */
  if(messageParam[0] != DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_EXPORT_RSA_KEY_OP_CODE)
  {
    error = DX_WRONG_OPCODE_FROM_SEP_ERR;
    goto end_function_unlock;
  }
   
  /* check the status */
  if(messageParam[1] != KMNG_OK)
  {
    error = messageParam[1];
    goto end_function_unlock;
  }
  
  /* set modulus size */
  *modulusSize_ptr = messageParam[2];
  
  /* read the modulus */
  error = SEPDriver_ReadParamater((DxUint32_t)modulus_ptr,
                          *modulusSize_ptr,
                          64*sizeof(DxUint32_t), 
                          &sramOffset , 
                          DX_TRUE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                          
  /* read E size */
  error = SEPDriver_ReadParamater((DxUint32_t)E_Size_ptr,
                          sizeof(DxUint32_t),
                          sizeof(DxUint32_t), 
                          &sramOffset , 
                          DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                          
  /* read the E pointer  */
  error = SEPDriver_ReadParamater((DxUint32_t)E_ptr,
                          *E_Size_ptr,
                          64*sizeof(DxUint32_t), 
                          &sramOffset , 
                          DX_TRUE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                          
  /* read D size */
  error = SEPDriver_ReadParamater((DxUint32_t)D_Size_ptr,
                          sizeof(DxUint32_t),
                          sizeof(DxUint32_t), 
                          &sramOffset , 
                          DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                          
  /* read the D pointer  */
  error = SEPDriver_ReadParamater((DxUint32_t)D_ptr,
                          *D_Size_ptr,
                          64*sizeof(DxUint32_t), 
                          &sramOffset , 
                          DX_TRUE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }

end_function_unlock:

  /* unlock access to the SEP */
  SEPDriver_Unlock();

end_function:

  return error;
}

/**
 * @brief     This function exports an DH key from key ring
 *
 *
 * @param[in] keyRingKeyData_ptr - pointer to the key data of the Key Ring
 * @param[out] keyConfiguration_ptr - configuration of the exported key
 * @param[in] exportKeyData_ptr - ipointer to the data of the key to be exported
 * @param[in] modulus_ptr - modulus pointer
 * @param[in] modulusSize_ptr - modulus size
 * @param[in] generator_ptr - generator pointer
 * @param[in] generatorSize_ptr - generator size
 * @param[in] E_ptr - E pointer
 * @param[in] E_Size_ptr - E size
 * @param[in] D_ptr - D pointer
 * @param[in] D_Size_ptr - D size
 * 
 * @return     DxError:  
 *                        
 */                                           
DxError_t _LLF_KMNG_ExportDHUserKey(KMNG_KeyDataParameter_t*   keyRingKeyData_ptr,
                                    KMNG_KeyConfiguration_t*   keyConfiguration_ptr,
                                    KMNG_KeyDataParameter_t*   exportKeyData_ptr,
                                    DxUint8_t*                 modulus_ptr,
                                    DxUint32_t*                modulusSize_ptr,
                                    DxUint8_t*                 generator_ptr,
                                    DxUint32_t*                generatorSize_ptr,
                                    DxUint8_t*                 E_ptr,
                                    DxUint32_t*                E_Size_ptr,
                                    DxUint8_t*                 D_ptr,
                                    DxUint32_t*                D_Size_ptr,
                                    DxUint32_t                 exportInternalFlag)
{
  /* sram offset */
  DxUint32_t  sramOffset;
  
  /* message param */
  DxUint32_t  messageParam[3];
  
  /* max length */
  DxUint32_t  maxLength;
  
  /* opcode */
  DxUint32_t  opcode;
  
  /* error */
  DxError_t   error;

  /*-------------------------------
      CODE
  ----------------------------------*/
  
  modulus_ptr = modulus_ptr;
  keyConfiguration_ptr = keyConfiguration_ptr;
  
  error = KMNG_OK;
  
   /* lock access to the SEP */
   error = SEPDriver_Lock();
   
   if(error != DX_OK)
   {
       goto end_function;
   }
  
  /*----------------------------
      start sending message to SEP 
  -----------------------------*/
  sramOffset = 0;
   
  /* start the message */
  SEPDriver_StartMessage(&sramOffset);
  
  /* prepare message  */
  if(exportInternalFlag == 0)
  {
  
    opcode = DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_EXPORT_DH_KEY_OP_CODE;
  }
  else
  {
    opcode = DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_READ_DH_KEY_OP_CODE;
  }
  
  messageParam[0] = opcode;
  
  /* send message */
  error = SEPDriver_WriteParamater((DxUint32_t)messageParam,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
 
  /* send key ring data */                   
  LLF_KMNG_SendKeyRingData(keyRingKeyData_ptr , &sramOffset);
  
  /* key password length */ 
  error = SEPDriver_WriteParamater((DxUint32_t)&exportKeyData_ptr->keyPasswordLen,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
  
  /* send key password */                         
  error = SEPDriver_WriteParamater((DxUint32_t)exportKeyData_ptr->keyPassword_ptr,
                           exportKeyData_ptr->keyPasswordLen,
                           64*sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                           
  /* send DH key data length */
  error = SEPDriver_WriteParamater((DxUint32_t)&exportKeyData_ptr->encryptedKeyDataLength,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
  
                           
  /* send DH key data */
  maxLength = ((exportKeyData_ptr->encryptedKeyDataLength + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
  error = SEPDriver_WriteParamater((DxUint32_t)exportKeyData_ptr->encryptedKeyData_ptr,
                           exportKeyData_ptr->encryptedKeyDataLength,
                           maxLength,
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                           
  SEPDriver_EndMessage(sramOffset);
            
  /* wait for the response */
  error = SEPDriver_POLL_FOR_REPONSE();
  if( error != DX_OK )
  {
    goto end_function_unlock;
  }
   
  /*-------------------
    start reading message from the SEP 
  ---------------------*/
   
  /* start the message */
  error = SEPDriver_StartIncomingMessage(&sramOffset);
  if( error != DX_OK )
  {
    goto end_function_unlock;
  }
   
  /* read opcode + status */
  error = SEPDriver_ReadParamater((DxUint32_t)messageParam,
                          sizeof(DxUint32_t) * 3,
                          sizeof(DxUint32_t) * 3, 
                          &sramOffset , 
                          DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
   
  /* check the opcode */
  if(messageParam[0] != opcode)
  {
    error = DX_WRONG_OPCODE_FROM_SEP_ERR;
    goto end_function_unlock;
  }
   
  /* check the status */
  if(messageParam[1] != KMNG_OK)
  {
    error = messageParam[1];
    goto end_function_unlock;
  }
  
  /* set modulus size in bytes */
  *modulusSize_ptr = messageParam[2];
  
  /* read the DH modulus */
  error = SEPDriver_ReadParamater((DxUint32_t)modulus_ptr,
                                  (*modulusSize_ptr),
                                  64*sizeof(DxUint32_t), 
                                  &sramOffset , 
                                  DX_TRUE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                          
  /* read generator size in bytes */
  error = SEPDriver_ReadParamater((DxUint32_t)generatorSize_ptr,
                                  sizeof(DxUint32_t),
                                  sizeof(DxUint32_t), 
                                  &sramOffset , 
                                  DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                          
  /* read the generator pointer  */
  error = SEPDriver_ReadParamater((DxUint32_t)generator_ptr,
                                  (*generatorSize_ptr),
                                  64*sizeof(DxUint32_t), 
                                  &sramOffset , 
                                  DX_TRUE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                          
  /* read N size */
  error = SEPDriver_ReadParamater((DxUint32_t)E_Size_ptr,
                                  sizeof(DxUint32_t),
                                  sizeof(DxUint32_t), 
                                  &sramOffset , 
                                  DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                          
  /* read the N pointer  */
  error = SEPDriver_ReadParamater((DxUint32_t)E_ptr,
                                  (*E_Size_ptr),
                                  64*sizeof(DxUint32_t), 
                                  &sramOffset , 
                                  DX_TRUE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
     
  if(exportInternalFlag == 0)
  {
    /* read D size */
    error = SEPDriver_ReadParamater((DxUint32_t)D_Size_ptr,
                                     sizeof(DxUint32_t),
                                     sizeof(DxUint32_t), 
                                     &sramOffset , 
                                     DX_FALSE);
    if(error != DX_OK)
    {
        goto end_function_unlock;
    }
                            
    /* read the D pointer  */
    error = SEPDriver_ReadParamater((DxUint32_t)D_ptr,
                                    (*D_Size_ptr),
                                    64*sizeof(DxUint32_t), 
                                    &sramOffset , 
                                    DX_TRUE);
    if(error != DX_OK)
    {
        goto end_function_unlock;
    }
  }

end_function_unlock:

  /* unlock access to the SEP */
  SEPDriver_Unlock();

end_function:

  return error;
}

/**
 * @brief     checks the authorization of the given key, by checking the hash of the password
 *            stored in the key data and the given password
 *
 * @param[in] keyRingKeyData_ptr - pointer to the key data of the Key Ring
 * @param[in] authorizeKeyData_ptr - pointer to the data of the key to be authorized
 * 
 * @return     DxError:  
 *                        
 */
DxError_t LLF_KMNG_AuthorizeKey(KMNG_KeyDataParameter_t*  keyRingKeyData_ptr,
                                KMNG_KeyDataParameter_t*  authorizeKeyData_ptr)
{
  /* sram offset */
  DxUint32_t  sramOffset;
  
  /* message param */
  DxUint32_t  messageParam[3];
  
  /* max length */
  DxUint32_t  maxLength;
  
  /* error */
  DxError_t   error;

  /*-------------------------------
      CODE
  ----------------------------------*/
  
  error = KMNG_OK;
  
   /* lock access to the SEP */
   error = SEPDriver_Lock();
   
   if(error != DX_OK)
   {
       goto end_function;
   }
  
  /*----------------------------
      start sending message to SEP 
  -----------------------------*/
  sramOffset = 0;
   
  /* start the message */
  SEPDriver_StartMessage(&sramOffset);
  
  /* prepare message  */
  messageParam[0] = DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_AUTHORIZE_KEY_OP_CODE;
 
  /* send message */
  error = SEPDriver_WriteParamater((DxUint32_t)messageParam,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
 
  /* send key ring data */                   
  LLF_KMNG_SendKeyRingData(keyRingKeyData_ptr , &sramOffset);
  
  /* key password length */ 
  error = SEPDriver_WriteParamater((DxUint32_t)&authorizeKeyData_ptr->keyPasswordLen,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
  
  /* send key password */                         
  error = SEPDriver_WriteParamater((DxUint32_t)authorizeKeyData_ptr->keyPassword_ptr,
                           authorizeKeyData_ptr->keyPasswordLen,
                           64*sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                           
  /* send  key data length */
  error = SEPDriver_WriteParamater((DxUint32_t)&authorizeKeyData_ptr->encryptedKeyDataLength,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
  
                           
  /* send  key data */
  maxLength = ((authorizeKeyData_ptr->encryptedKeyDataLength + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
  error = SEPDriver_WriteParamater((DxUint32_t)authorizeKeyData_ptr->encryptedKeyData_ptr,
                           authorizeKeyData_ptr->encryptedKeyDataLength,
                           maxLength,
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                           
  SEPDriver_EndMessage(sramOffset);
            
  /* wait for the response */
  error = SEPDriver_POLL_FOR_REPONSE();
  if( error != DX_OK )
  {
    goto end_function_unlock;
  }
   
  /*-------------------
    start reading message from the SEP 
  ---------------------*/
   
  /* start the message */
  error = SEPDriver_StartIncomingMessage(&sramOffset);
  if( error != DX_OK )
  {
    goto end_function_unlock;
  }
   
  /* read opcode + status */
  error = SEPDriver_ReadParamater((DxUint32_t)messageParam,
                          sizeof(DxUint32_t) * 2,
                          sizeof(DxUint32_t) * 2, 
                          &sramOffset , 
                          DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
   
  /* check the opcode */
  if(messageParam[0] != DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_AUTHORIZE_KEY_OP_CODE)
  {
    error = DX_WRONG_OPCODE_FROM_SEP_ERR;
    
  }
  
  error = messageParam[1];

end_function_unlock:   

  /* unlock access to the SEP */
  SEPDriver_Unlock();
  
end_function:
  
  return error;
}
                                                                          
                                       
/**
 * @brief     This function transfers key from one key ring into another (actually decrypts it and then enctypts it with destination key ring key).
 *
 * @param[in] srcKeyRingKeyData_ptr - pointer to the key data of the source Key Ring
 * @param[in] srcKeyData_ptr - key data of the source key to be transfered
 * @param[in] destRingKeyData_ptr - pointer to the key data of the destination Key Ring
 * @param[out] destKeyData_ptr - data of key transferred to the destination key ring
 *
 * @return     DxError:  
 *                        
 */                                
DxError_t LLF_KMNG_TransferKey(KMNG_KeyDataParameter_t*  srcKeyRingKeyData_ptr,
                               KMNG_KeyDataParameter_t*  srcKeyData_ptr,
                               KMNG_KeyDataParameter_t*  destRingKeyData_ptr,
                               KMNG_KeyDataParameter_t*  destKeyData_ptr)
{
  /* sram offset */
  DxUint32_t  sramOffset;
  
  /* message param */
  DxUint32_t  messageParam[3];
  
  /* max length */
  DxUint32_t  maxLength;
  
  /* error */
  DxError_t   error;

  /*-------------------------------
      CODE
  ----------------------------------*/

  error = KMNG_OK;

   /* lock access to the SEP */
   error = SEPDriver_Lock();
   
   if(error != DX_OK)
   {
       goto end_function;
   }
  
  /*----------------------------
      start sending message to SEP 
  -----------------------------*/
  sramOffset = 0;
   
  /* start the message */
  SEPDriver_StartMessage(&sramOffset);
  
  /* prepare message  */
  messageParam[0] = DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_TRANSFER_KEY_OP_CODE;
 
  /* send message */
  error = SEPDriver_WriteParamater((DxUint32_t)messageParam,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
 
  /* send src key ring data */                   
  LLF_KMNG_SendKeyRingData(srcKeyRingKeyData_ptr , &sramOffset);
  
  /* src key password length */ 
  error = SEPDriver_WriteParamater((DxUint32_t)&srcKeyData_ptr->keyPasswordLen,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
  
  /* send src key password */                         
  error = SEPDriver_WriteParamater((DxUint32_t)srcKeyData_ptr->keyPassword_ptr,
                           srcKeyData_ptr->keyPasswordLen,
                           64*sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                                                      
  /* send destination key ring data */
  LLF_KMNG_SendKeyRingData(destRingKeyData_ptr , &sramOffset);
  
                           
  
  /* dest key password length */ 
  error = SEPDriver_WriteParamater((DxUint32_t)&destKeyData_ptr->keyPasswordLen,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
  
  /* send dest key password */                         
  error = SEPDriver_WriteParamater((DxUint32_t)destKeyData_ptr->keyPassword_ptr,
                           destKeyData_ptr->keyPasswordLen,
                           64*sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }

  /* send  src key data length */
  error = SEPDriver_WriteParamater((DxUint32_t)&srcKeyData_ptr->encryptedKeyDataLength,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
  
                           
  /* send  src key data */
  maxLength = ((srcKeyData_ptr->encryptedKeyDataLength + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
  error = SEPDriver_WriteParamater((DxUint32_t)srcKeyData_ptr->encryptedKeyData_ptr,
                           srcKeyData_ptr->encryptedKeyDataLength,
                           maxLength,
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                           
  SEPDriver_EndMessage(sramOffset);
            
  /* wait for the response */
  error = SEPDriver_POLL_FOR_REPONSE();
  if( error != DX_OK )
  {
    goto end_function_unlock;
  }
   
  /*-------------------
    start reading message from the SEP 
  ---------------------*/
   
  /* start the message */
  error = SEPDriver_StartIncomingMessage(&sramOffset);
  if( error != DX_OK )
  {
    goto end_function_unlock;
  }
   
  /* read opcode + status */
  error = SEPDriver_ReadParamater((DxUint32_t)messageParam,
                          sizeof(DxUint32_t) * 3,
                          sizeof(DxUint32_t) * 3, 
                          &sramOffset , 
                          DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
   
  /* check the opcode */
  if(messageParam[0] != DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_TRANSFER_KEY_OP_CODE)
  {
    error = DX_WRONG_OPCODE_FROM_SEP_ERR;
    
  }
  
  /* check the status */
  if(messageParam[1] != KMNG_OK)
  {
    error = messageParam[1];
    goto end_function_unlock;
  }
  
  /* set encryption data length */
  destKeyData_ptr->encryptedKeyDataLength = messageParam[2];
  
  /* read the encrypted data */
  maxLength = ((destKeyData_ptr->encryptedKeyDataLength + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
  error = SEPDriver_ReadParamater((DxUint32_t)destKeyData_ptr->encryptedKeyData_ptr,
                          destKeyData_ptr->encryptedKeyDataLength,
                          maxLength, 
                          &sramOffset , 
                          DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }

end_function_unlock:
                           
  /* unlock access to the SEP */
  SEPDriver_Unlock();

end_function:  
  
  return error;
}
                                
                                
/**
 * @brief     This function prepares the AES contest for CRYS function usage based on the key ring AES key
 *
 * @param[in] keyRingKeyData_ptr - pointer to the key of the Key Ring
 * @param[in] keyData_ptr - pointer to the key to be set into context
 * @param[out]aesContext_ptr - memory to stores created AES context  
 * 
 * @return     DxError:  
 *                        
 */
DxError_t LLF_KMNG_ActivateAESKey(KMNG_KeyDataParameter_t*    keyRingKeyData_ptr,
                                      KMNG_KeyDataParameter_t*    keyData_ptr,
                                      DxUint8_t*                  aesContext_ptr)
{
  /* sram offset */
  DxUint32_t  sramOffset;
  
  /* message param */
  DxUint32_t  messageParam[3];
  
  /* max length */
  DxUint32_t  maxLength;
  
  /* error */
  DxError_t   error;

  /*-------------------------------
      CODE
  ----------------------------------*/
  
  error = KMNG_OK;

   /* lock access to the SEP */
   error = SEPDriver_Lock();
   
   if(error != DX_OK)
   {
       goto end_function;
   }
     
  /*----------------------------
      start sending message to SEP 
  -----------------------------*/
  sramOffset = 0;
   
  /* start the message */
  SEPDriver_StartMessage(&sramOffset);
  
  /* prepare message  */
  messageParam[0] = DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_ACTIVATE_AES_KEY_OP_CODE;
 
  /* send message */
  error = SEPDriver_WriteParamater((DxUint32_t)messageParam,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
 
  /* send key ring data */                   
  LLF_KMNG_SendKeyRingData(keyRingKeyData_ptr , &sramOffset);
  
  /* key password length */ 
  error = SEPDriver_WriteParamater((DxUint32_t)&keyData_ptr->keyPasswordLen,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
  
  /* send key password */                         
  error = SEPDriver_WriteParamater((DxUint32_t)keyData_ptr->keyPassword_ptr,
                           keyData_ptr->keyPasswordLen,
                           64*sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                           
  /* send key data length */
  error = SEPDriver_WriteParamater((DxUint32_t)&keyData_ptr->encryptedKeyDataLength,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
  
                           
  /* send key data */
  maxLength = ((keyData_ptr->encryptedKeyDataLength + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
  error = SEPDriver_WriteParamater((DxUint32_t)keyData_ptr->encryptedKeyData_ptr,
                           keyData_ptr->encryptedKeyDataLength,
                           maxLength,
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                           
  SEPDriver_EndMessage(sramOffset);
            
  /* wait for the response */
  error = SEPDriver_POLL_FOR_REPONSE();
  if( error != DX_OK )
  {
    goto end_function_unlock;
  }
   
  /*-------------------
    start reading message from the SEP 
  ---------------------*/
   
  /* start the message */
  error = SEPDriver_StartIncomingMessage(&sramOffset);
  if( error != DX_OK )
  {
    goto end_function_unlock;
  }
   
  /* read opcode + status */
  error = SEPDriver_ReadParamater((DxUint32_t)messageParam,
                          sizeof(DxUint32_t) * 3,
                          sizeof(DxUint32_t) * 3, 
                          &sramOffset , 
                          DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
   
  /* check the opcode */
  if(messageParam[0] != DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_ACTIVATE_AES_KEY_OP_CODE)
  {
    error = DX_WRONG_OPCODE_FROM_SEP_ERR;
    
  }
  
  /* check the status */
  if(messageParam[1] != KMNG_OK)
  {
    error = messageParam[1];
    goto end_function_unlock;
  }
  
  /* read the context */
  maxLength = ((messageParam[2] + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
  error = SEPDriver_ReadParamater((DxUint32_t)aesContext_ptr,
                          messageParam[2],
                          maxLength, 
                          &sramOffset , 
                          DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }

end_function_unlock:
                           
  /* unlock access to the SEP */
  SEPDriver_Unlock();

end_function:  
  
  return error;
}

/**
 * @brief     This function prepares the DES contest for CRYS function usage based on the key ring AES key
 *
 * @param[in] keyRingKeyData_ptr - pointer to the key of the Key Ring
 * @param[in] keyData_ptr - pointer to the key to be set into context
 * @param[out]aesContext_ptr - memory to stores created AES context  
 * 
 * @return     DxError:  
 *                        
 */
DxError_t LLF_KMNG_ActivateDESKey(KMNG_KeyDataParameter_t*  keyRingKeyData_ptr,
                                  KMNG_KeyDataParameter_t*  keyData_ptr,
                                  KMNG_DES_WrappedKey_t     DES_WrappedKey_ptr)
{
  /* sram offset */
  DxUint32_t  sramOffset;
  
  /* message param */
  DxUint32_t  messageParam[3];
  
  /* max length */
  DxUint32_t  maxLength;
  
  /* error */
  DxError_t   error;

  /*-------------------------------
      CODE
  ----------------------------------*/
  
  error = KMNG_OK;

   /* lock access to the SEP */
   error = SEPDriver_Lock();
   
   if(error != DX_OK)
   {
       goto end_function;
   }
     
  /*----------------------------
      start sending message to SEP 
  -----------------------------*/
  sramOffset = 0;
   
  /* start the message */
  SEPDriver_StartMessage(&sramOffset);
  
  /* prepare message  */
  messageParam[0] = DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_ACTIVATE_DES_KEY_OP_CODE;
 
  /* send message */
  error = SEPDriver_WriteParamater((DxUint32_t)messageParam,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
 
  /* send key ring data */                   
  LLF_KMNG_SendKeyRingData(keyRingKeyData_ptr , &sramOffset);
  
  /* key password length */ 
  error = SEPDriver_WriteParamater((DxUint32_t)&keyData_ptr->keyPasswordLen,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
  
  /* send key password */                         
  error = SEPDriver_WriteParamater((DxUint32_t)keyData_ptr->keyPassword_ptr,
                           keyData_ptr->keyPasswordLen,
                           64*sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                           
  /* send key data length */
  error = SEPDriver_WriteParamater((DxUint32_t)&keyData_ptr->encryptedKeyDataLength,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
  
                           
  /* send key data */
  maxLength = ((keyData_ptr->encryptedKeyDataLength + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
  error = SEPDriver_WriteParamater((DxUint32_t)keyData_ptr->encryptedKeyData_ptr,
                           keyData_ptr->encryptedKeyDataLength,
                           maxLength,
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                           
  SEPDriver_EndMessage(sramOffset);
            
  /* wait for the response */
  error = SEPDriver_POLL_FOR_REPONSE();
  if( error != DX_OK )
  {
    goto end_function_unlock;
  }
   
  /*-------------------
    start reading message from the SEP 
  ---------------------*/
   
  /* start the message */
  error = SEPDriver_StartIncomingMessage(&sramOffset);
  if( error != DX_OK )
  {
    goto end_function_unlock;
  }
   
  /* read opcode + status */
  error = SEPDriver_ReadParamater((DxUint32_t)messageParam,
                          sizeof(DxUint32_t) * 3,
                          sizeof(DxUint32_t) * 3, 
                          &sramOffset , 
                          DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
   
  /* check the opcode */
  if(messageParam[0] != DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_ACTIVATE_DES_KEY_OP_CODE)
  {
    error = DX_WRONG_OPCODE_FROM_SEP_ERR;
    
  }
  
  /* check the status */
  if(messageParam[1] != KMNG_OK)
  {
    error = messageParam[1];
    goto end_function_unlock;
  }
  
  /* read the context */
  maxLength = ((messageParam[2] + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
  error = SEPDriver_ReadParamater((DxUint32_t)DES_WrappedKey_ptr,
                          messageParam[2],
                          maxLength, 
                          &sramOffset , 
                          DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }

end_function_unlock:
                           
  /* unlock access to the SEP */
  SEPDriver_Unlock();
  
end_function:
  
  return error;
}
                                    
/**
 * @brief     This function prepares the HMAC contest for CRYS function usage based on the key ring HMAC key
 *
 * @param[in] keyRingKeyData_ptr - pointer to the key of the Key Ring
 * @param[in] keyData_ptr - pointer to the key to be set into context
 * @param[out]hmacContext_ptr - memory to stores created HMAC context  
 * 
 * @return     DxError:  
 *                        
 */
DxError_t LLF_KMNG_ActivateHMACKey(KMNG_KeyDataParameter_t*   keyRingKeyData_ptr,
                                       KMNG_KeyDataParameter_t*   keyData_ptr,
                                       KMNG_HMAC_WrappedKey_t     HMAC_WrappedKey_ptr)
{
  /* sram offset */
  DxUint32_t  sramOffset;
  
  /* message param */
  DxUint32_t  messageParam[3];
  
  /* max length */
  DxUint32_t  maxLength;
  
  /* error */
  DxError_t   error;

  /*-------------------------------
      CODE
  ----------------------------------*/

  error = KMNG_OK;
  
   /* lock access to the SEP */
   error = SEPDriver_Lock();
   
   if(error != DX_OK)
   {
       goto end_function;
   }
     
  /*----------------------------
      start sending message to SEP 
  -----------------------------*/
  sramOffset = 0;
   
  /* start the message */
  SEPDriver_StartMessage(&sramOffset);
  
  /* prepare message  */
  messageParam[0] = DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_ACTIVATE_HMAC_KEY_OP_CODE;
 
  /* send message */
  error = SEPDriver_WriteParamater((DxUint32_t)messageParam,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
 
  /* send key ring data */                   
  LLF_KMNG_SendKeyRingData(keyRingKeyData_ptr , &sramOffset);
  
  /* key password length */ 
  error = SEPDriver_WriteParamater((DxUint32_t)&keyData_ptr->keyPasswordLen,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
  
  /* send key password */                         
  error = SEPDriver_WriteParamater((DxUint32_t)keyData_ptr->keyPassword_ptr,
                           keyData_ptr->keyPasswordLen,
                           64*sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                           
  /* send key data length */
  error = SEPDriver_WriteParamater((DxUint32_t)&keyData_ptr->encryptedKeyDataLength,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
  
                           
  /* send key data */
  maxLength = ((keyData_ptr->encryptedKeyDataLength + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
  error = SEPDriver_WriteParamater((DxUint32_t)keyData_ptr->encryptedKeyData_ptr,
                           keyData_ptr->encryptedKeyDataLength,
                           maxLength,
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                           
  SEPDriver_EndMessage(sramOffset);
            
  /* wait for the response */
  error = SEPDriver_POLL_FOR_REPONSE();
  if( error != DX_OK )
  {
    goto end_function_unlock;
  }
   
  /*-------------------
    start reading message from the SEP 
  ---------------------*/
   
  /* start the message */
  error = SEPDriver_StartIncomingMessage(&sramOffset);
  if( error != DX_OK )
  {
    goto end_function_unlock;
  }
   
  /* read opcode + status */
  error = SEPDriver_ReadParamater((DxUint32_t)messageParam,
                          sizeof(DxUint32_t) * 3,
                          sizeof(DxUint32_t) * 3, 
                          &sramOffset , 
                          DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
   
  /* check the opcode */
  if(messageParam[0] != DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_ACTIVATE_HMAC_KEY_OP_CODE)
  {
    error = DX_WRONG_OPCODE_FROM_SEP_ERR;
    
  }
  
  /* check the status */
  if(messageParam[1] != KMNG_OK)
  {
    error = messageParam[1];
    goto end_function_unlock;
  }
  
  /* read the context */
  maxLength = ((messageParam[2] + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
  error = SEPDriver_ReadParamater((DxUint32_t)HMAC_WrappedKey_ptr,
                          messageParam[2],
                          maxLength, 
                          &sramOffset , 
                          DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }

end_function_unlock:
                           
  /* unlock access to the SEP */
  SEPDriver_Unlock();
  
end_function:
  
  return error;
}
                                     
/**
 * @brief     This function prepares the RSA contest for CRYS function usage based on the key ring RSA key
 *
 * @param[in] keyRingKeyData_ptr - pointer to the key of the Key Ring
 * @param[in] keyData_ptr - pointer to the key to be set into context
 * @param[out]rsaContext_ptr - memory to stores created RSA context  
 * 
 * @return     DxError:  
 *                        
 */
DxError_t LLF_KMNG_ActivateRSAKey(KMNG_KeyDataParameter_t*    keyRingKeyData_ptr,
                                      KMNG_KeyDataParameter_t*    keyData_ptr,
                                      KMNG_RsaKeyOperationType_t  rsaOperationType,
                                      DxUint8_t*                  RSA_WrappedKey_ptr,
                                      DxUint32_t*                  RSA_WrappedKeyLen_ptr)
{
  /* sram offset */
  DxUint32_t  sramOffset;
  
  /* message param */
  DxUint32_t  messageParam[3];
  
  /* max length */
  DxUint32_t  maxLength;
  
  /* error */
  DxError_t   error;

  /*-------------------------------
      CODE
  ----------------------------------*/
  
  error = KMNG_OK;

   /* lock access to the SEP */
   error = SEPDriver_Lock();
   
   if(error != DX_OK)
   {
       goto end_function;
   }
     
  /*----------------------------
      start sending message to SEP 
  -----------------------------*/
  sramOffset = 0;
   
  /* start the message */
  SEPDriver_StartMessage(&sramOffset);
  
  /* prepare message  */
  messageParam[0] = DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_ACTIVATE_RSA_KEY_OP_CODE;
 
  /* send message */
  error = SEPDriver_WriteParamater((DxUint32_t)messageParam,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
 
  /* send key ring data */                   
  LLF_KMNG_SendKeyRingData(keyRingKeyData_ptr , &sramOffset);
  
  /* key password length */ 
  error = SEPDriver_WriteParamater((DxUint32_t)&keyData_ptr->keyPasswordLen,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
  
  /* send key password */                         
  error = SEPDriver_WriteParamater((DxUint32_t)keyData_ptr->keyPassword_ptr,
                           keyData_ptr->keyPasswordLen,
                           64*sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                           
  /* send key type  */
  error = SEPDriver_WriteParamater((DxUint32_t)&rsaOperationType,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                           
  /* send key data length */
  error = SEPDriver_WriteParamater((DxUint32_t)&keyData_ptr->encryptedKeyDataLength,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
  
                           
  /* send key data */
  maxLength = ((keyData_ptr->encryptedKeyDataLength + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
  error = SEPDriver_WriteParamater((DxUint32_t)keyData_ptr->encryptedKeyData_ptr,
                           keyData_ptr->encryptedKeyDataLength,
                           maxLength,
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                           
  SEPDriver_EndMessage(sramOffset);
            
  /* wait for the response */
  error = SEPDriver_POLL_FOR_REPONSE();
  if( error != DX_OK )
  {
    goto end_function_unlock;
  }
   
  /*-------------------
    start reading message from the SEP 
  ---------------------*/
   
  /* start the message */
  error = SEPDriver_StartIncomingMessage(&sramOffset);
  if( error != DX_OK )
  {
    goto end_function_unlock;
  }
   
  /* read opcode + status */
  error = SEPDriver_ReadParamater((DxUint32_t)messageParam,
                          sizeof(DxUint32_t) * 3,
                          sizeof(DxUint32_t) * 3, 
                          &sramOffset , 
                          DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
   
  /* check the opcode */
  if(messageParam[0] != DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_ACTIVATE_RSA_KEY_OP_CODE)
  {
    error = DX_WRONG_OPCODE_FROM_SEP_ERR;
    
  }
  
  /* check the status */
  if(messageParam[1] != KMNG_OK)
  {
    error = messageParam[1];
    goto end_function_unlock;
  }

  *RSA_WrappedKeyLen_ptr = messageParam[2];

  /* read the context */
  maxLength = ((messageParam[2] + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
  error = SEPDriver_ReadParamater((DxUint32_t)RSA_WrappedKey_ptr,
                          messageParam[2],
                          maxLength, 
                          &sramOffset , 
                          DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }

end_function_unlock:
                           
  /* unlock access to the SEP */
  SEPDriver_Unlock();
  
end_function:
  
  return error;
}
                                    
                                    
/**
 * @brief     This function prepares the DH contest for CRYS function usage based on the key ring DH key
 *
 * @param[in] keyRingKeyData_ptr - pointer to the key of the Key Ring
 * @param[in] keyData_ptr - pointer to the key to be set into context
 * @param[out]dhContext_ptr - memory to stores created DH context  
 * @param[out]DH_WrappedKeyLen_ptr - the length of the wraped key 
 * 
 * @return     DxError:  
 *                        
 */
DxError_t LLF_KMNG_ActivateDHKey(KMNG_KeyDataParameter_t*   keyRingKeyData_ptr,
                								 KMNG_KeyDataParameter_t*   keyData_ptr,
                								 KMNG_DH_WrappedKey_t		    DH_WrappedKey_ptr,
                								 DxUint32_t*				        DH_WrappedKeyLen_ptr)
{
  /* sram offset */
  DxUint32_t  sramOffset;
  
  /* message param */
  DxUint32_t  messageParam[3];
  
  /* max length */
  DxUint32_t  maxLength;
  
  /* error */
  DxError_t   error;

  /*-------------------------------
      CODE
  ----------------------------------*/
  
  error = KMNG_OK;

   /* lock access to the SEP */
   error = SEPDriver_Lock();
   
   if(error != DX_OK)
   {
       goto end_function;
   }
     
  /*----------------------------
      start sending message to SEP 
  -----------------------------*/
  sramOffset = 0;
   
  /* start the message */
  SEPDriver_StartMessage(&sramOffset);
  
  /* prepare message  */
  messageParam[0] = DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_ACTIVATE_DH_KEY_OP_CODE;
 
  /* send message */
  error = SEPDriver_WriteParamater((DxUint32_t)messageParam,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
 
  /* send key ring data */                   
  LLF_KMNG_SendKeyRingData(keyRingKeyData_ptr , &sramOffset);
  
  /* key password length */ 
  error = SEPDriver_WriteParamater((DxUint32_t)&keyData_ptr->keyPasswordLen,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
  
  /* send key password */                         
  error = SEPDriver_WriteParamater((DxUint32_t)keyData_ptr->keyPassword_ptr,
                           keyData_ptr->keyPasswordLen,
                           64*sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                           
  /* send key data length */
  error = SEPDriver_WriteParamater((DxUint32_t)&keyData_ptr->encryptedKeyDataLength,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
  
                           
  /* send key data */
  maxLength = ((keyData_ptr->encryptedKeyDataLength + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
  error = SEPDriver_WriteParamater((DxUint32_t)keyData_ptr->encryptedKeyData_ptr,
                           keyData_ptr->encryptedKeyDataLength,
                           maxLength,
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                           
  SEPDriver_EndMessage(sramOffset);
            
  /* wait for the response */
  error = SEPDriver_POLL_FOR_REPONSE();
  if( error != DX_OK )
  {
    goto end_function_unlock;
  }
   
  /*-------------------
    start reading message from the SEP 
  ---------------------*/
   
  /* start the message */
  error = SEPDriver_StartIncomingMessage(&sramOffset);
  if( error != DX_OK )
  {
    goto end_function_unlock;
  }
   
  /* read opcode + status */
  error = SEPDriver_ReadParamater((DxUint32_t)messageParam,
                          sizeof(DxUint32_t) * 3,
                          sizeof(DxUint32_t) * 3, 
                          &sramOffset , 
                          DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
   
  /* check the opcode */
  if(messageParam[0] != DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_ACTIVATE_DH_KEY_OP_CODE)
  {
    error = DX_WRONG_OPCODE_FROM_SEP_ERR;
    
  }
  
  /* check the status */
  if(messageParam[1] != KMNG_OK)
  {
    error = messageParam[1];
    goto end_function_unlock;
  }
  
  /* read the context */
  maxLength = ((messageParam[2] + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
  error = SEPDriver_ReadParamater((DxUint32_t)DH_WrappedKey_ptr,
                          messageParam[2],
                          maxLength, 
                          &sramOffset , 
                          DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }

end_function_unlock:
                           
  /* unlock access to the SEP */
  SEPDriver_Unlock();
  
end_function:
  
  return error;
}


/*************************************************************************************************/
/*  LLF_KMNG_CalcKeyRingAesMAC ()                                                                */ 
/*************************************************************************************************/

DxError_t LLF_KMNG_CalcKeyRingAesMAC(KMNG_KeyDataParameter_t* keyRingDataParam_ptr,
                  									 DxUint8_t*               keyRing_ptr,
                  									 DxUint32_t               keyRingLength,
                  									 DxUint8_t*               macBuffer_ptr)
{
  /* sram offset */
  DxUint32_t          sramOffset;
   
  /* address of the first table in the input list */
  DxUint32_t          firstInputLLITableAddr;
   
  /* number of entries in the first input tables */
  DxUint32_t          numInTableEntries;
  
  /* the size of data in the first table */
  DxUint32_t          firstTableSize;
  
  /* message param */
  DxUint32_t          messageParam[6];
  
  /* error */
  DxError_t           error;

  /*-------------------------------
      CODE
  ----------------------------------*/
  
  error = KMNG_OK;

   /* lock access to the SEP */
   error = SEPDriver_Lock();
   
   if(error != DX_OK)
   {
       goto end_function;
   }
   
  /* prepare LLI table */
  error = SEPDriver_PREPARE_INPUT_SYM_DMA_TABLE((DxUint32_t)keyRing_ptr,
                                               keyRingLength,
                                               CRYS_AES_BLOCK_SIZE_IN_BYTES,
                                               &firstInputLLITableAddr,
                                               &numInTableEntries,
                                               &firstTableSize);
  if(error != KMNG_OK)
  {
    goto end_function_unlock;
  }
  
  /*----------------------------
      start sending message to SEP 
  -----------------------------*/
  sramOffset = 0;
   
  /* start the message */
  SEPDriver_StartMessage(&sramOffset);
  
  /* prepare message  */
  messageParam[0] = DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_CALC_KEY_RING_MAC_OP_CODE;
  
  /* send message */
  error = SEPDriver_WriteParamater((DxUint32_t)messageParam,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
 
  /* send key ring data */                   
  LLF_KMNG_SendKeyRingData(keyRingDataParam_ptr , &sramOffset);
  
  messageParam[0] = firstInputLLITableAddr;
  messageParam[1] = numInTableEntries;
  messageParam[2] = 0x0;/*NO OUTPUT in AES-MAC operation*/
  messageParam[3] = 0x0;/*NO OUTPUT in AES-MAC operation*/
  messageParam[4] = firstTableSize;
  
  /* send message */
  error = SEPDriver_WriteParamater((DxUint32_t)messageParam,
                           sizeof(DxUint32_t) * 5,
                           sizeof(DxUint32_t) * 5,
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
 
                           
  SEPDriver_EndMessage(sramOffset);
            
  /* wait for the response */
  error = SEPDriver_POLL_FOR_REPONSE();
  if( error != DX_OK )
  {
    goto end_function_unlock;
  }
  
  /* free the tables */
  error = SEPDriver_FreeDMATables();
  if(error)
  {
    goto end_function_unlock;
  }
   
  /*-------------------
    start reading message from the SEP 
  ---------------------*/
   
  /* start the message */
  error = SEPDriver_StartIncomingMessage(&sramOffset);
  if( error != DX_OK )
  {
    goto end_function_unlock;
  }
   
  /* read opcode + status */
  error = SEPDriver_ReadParamater((DxUint32_t)messageParam,
                          sizeof(DxUint32_t) * 2,
                          sizeof(DxUint32_t) * 2, 
                          &sramOffset , 
                          DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
   
  /* check the opcode */
  if(messageParam[0] != DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_CALC_KEY_RING_MAC_OP_CODE)
  {
    error = DX_WRONG_OPCODE_FROM_SEP_ERR;
    
  }
  
  /* check the status */
  if(messageParam[1] != KMNG_OK)
  {
    error = messageParam[1];
    goto end_function_unlock;
  }
  
  /* read the AES MAC context */
  error = SEPDriver_ReadParamater((DxUint32_t)macBuffer_ptr,
                           4 * sizeof(DxUint32_t),
                           4 * sizeof(DxUint32_t),
                           &sramOffset,
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }

end_function_unlock:
                           
  /* unlock access to the SEP */
  SEPDriver_Unlock();
  
end_function:
  
  return error;
  
}


/**
 * @brief     this fucntion caluclates the AES mac on the given image buffer and compares it to the given MAC buffer 
 *
 * @param[in] keyRingKeyData_ptr - pointer to the key of the Key Ring
 * @param[in] imageBuffer_ptr - pointer to the image to be calculate on
 * @param[in] imageBufferSize - image buffer size
 * @param[out]MacBuffer_ptr - The AES MAC of the keyRing_ptr data  
 * 
 * @return     DxError:  
 *                        
 */
DxError_t LLF_KMNG_CalcAndVerifyKeyRingAesMAC(KMNG_KeyDataParameter_t *keyRingKeyData_ptr,
                      											  DxUint8_t			*imageBuffer_ptr,
                      											  DxUint32_t		imageBufferSize,
                      											  DxUint8_t			*MacBuffer_ptr)
{
  /* sram offset */
  DxUint32_t      sramOffset;
   
  /* address of the first table in the input list */
  DxUint32_t      firstInputLLITableAddr;
   
  /* number of entries in the first input tables */
  DxUint32_t      numInTableEntries;
  
  /* size of data in the first table */
  DxUint32_t      firstTableSize;
  
  /* message param */
  DxUint32_t      messageParam[6];
  
  /* error */
  DxError_t       error;

  /*-------------------------------
      CODE
  ----------------------------------*/
  
  error = KMNG_OK;

   /* lock access to the SEP */
   error = SEPDriver_Lock();
   
   if(error != DX_OK)
   {
       goto end_function;
   }

  /* prepare LLI table */
  error = SEPDriver_PREPARE_INPUT_SYM_DMA_TABLE((DxUint32_t)imageBuffer_ptr,
                                                imageBufferSize,
                                                CRYS_AES_BLOCK_SIZE_IN_BYTES,
                                                &firstInputLLITableAddr,
                                                &numInTableEntries,
                                                &firstTableSize);  
  if(error != KMNG_OK)
  {
    goto end_function_unlock;
  }
  
  /*----------------------------
      start sending message to SEP 
  -----------------------------*/
  sramOffset = 0;
   
  /* start the message */
  SEPDriver_StartMessage(&sramOffset);
  
  /* prepare message  */
  messageParam[0] = DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_CALC_AND_VERIFY_KEY_RING_MAC_OP_CODE;
  messageParam[1] = firstInputLLITableAddr;
  messageParam[2] = numInTableEntries;
  messageParam[3] = 0x0;/*NO OUTPUT in AES-MAC operation*/
  messageParam[4] = 0x0;/*NO OUTPUT in AES-MAC operation*/
  messageParam[5] = firstTableSize;
  
  /* send message */
  error = SEPDriver_WriteParamater((DxUint32_t)messageParam,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
 
  /* send key ring data */                   
  LLF_KMNG_SendKeyRingData(keyRingKeyData_ptr , &sramOffset);


  messageParam[0] = firstInputLLITableAddr;
  messageParam[1] = numInTableEntries;
  messageParam[2] = 0x0;/*NO OUTPUT in AES-MAC operation*/
  messageParam[3] = 0x0;/*NO OUTPUT in AES-MAC operation*/
  messageParam[4] = imageBufferSize;
  
  /* send message */
  error = SEPDriver_WriteParamater((DxUint32_t)messageParam,
                           sizeof(DxUint32_t) * 5,
                           sizeof(DxUint32_t) * 5,
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }

  /* send the MAC to compare */
  error = SEPDriver_WriteParamater((DxUint32_t)MacBuffer_ptr,
                           4 * sizeof(DxUint32_t),
                           4 * sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
  
                           
  SEPDriver_EndMessage(sramOffset);
            
  /* wait for the response */
  error = SEPDriver_POLL_FOR_REPONSE();
  if( error != DX_OK )
  {
    goto end_function_unlock;
  }
  
  /* free the tables */
  error = SEPDriver_FreeDMATables();
  if(error)
  {
    goto end_function_unlock;
  }
   
  /*-------------------
    start reading message from the SEP 
  ---------------------*/
   
  /* start the message */
  error = SEPDriver_StartIncomingMessage(&sramOffset);
  if( error != DX_OK )
  {
    goto end_function_unlock;
  }
   
  /* read opcode + status */
  error = SEPDriver_ReadParamater((DxUint32_t)messageParam,
                          sizeof(DxUint32_t) * 2,
                          sizeof(DxUint32_t) * 2, 
                          &sramOffset , 
                          DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
   
  /* check the opcode */
  if(messageParam[0] != DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_CALC_AND_VERIFY_KEY_RING_MAC_OP_CODE)
  {
    error = DX_WRONG_OPCODE_FROM_SEP_ERR;
    
  }
  
  /* check the status */
  if(messageParam[1] != KMNG_OK)
  {
    error = messageParam[1];
    goto end_function_unlock;
  }

end_function_unlock:
                           
  /* unlock access to the SEP */
  SEPDriver_Unlock();

end_function:  
  
  return error;
}

/*--------------------------------------
    Private Functions
----------------------------------------*/


/*************************************************************************************************/
/*  LLF_KMNG_SendCreateKeyRingMessage ()                                                         */ 
/*************************************************************************************************/
static DxError_t LLF_KMNG_SendCreateKeyRingMessage(KMNG_KeyDataParameter_t*   newKeyData_ptr,
                                                  KMNG_KeyConfiguration_t*   keyConfiguration_ptr)
{
  /* sram offset */
  DxUint32_t  sramOffset;
  
  /* message param */
  DxUint32_t  messageParam[3];
  
  /* max length */
  DxUint32_t  maxLength;
  
  /* error */
  DxError_t   error;

  /*-------------------------------
      CODE
  ----------------------------------*/
  
  error = KMNG_OK;
  
   /* lock access to the SEP */
   error = SEPDriver_Lock();
   
   if(error != DX_OK)
   {
       goto end_function;
   }
  
  /*----------------------------
      start sending message to SEP 
  -----------------------------*/
  sramOffset = 0;
   
  /* start the message */
  SEPDriver_StartMessage(&sramOffset);
  

    
  /* prepare message  */
  messageParam[0] = DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_CREATE_KEY_RING_OP_CODE;
  messageParam[1] = newKeyData_ptr->keyPasswordLen;

  /* send message */
  error = SEPDriver_WriteParamater((DxUint32_t)messageParam,
                           sizeof(DxUint32_t) * 2,
                           sizeof(DxUint32_t) * 2,
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
 
                             
  /* send  key password */
  error = SEPDriver_WriteParamater((DxUint32_t)newKeyData_ptr->keyPassword_ptr,
                             newKeyData_ptr->keyPasswordLen,
                             64*sizeof(DxUint32_t),
                             &sramOffset , 
                             DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
    
  /* send  key configuration */
  error = SEPDriver_WriteParamater((DxUint32_t)keyConfiguration_ptr,
                            sizeof(DxUint32_t) * 6,
                            sizeof(DxUint32_t) * 6,
                            &sramOffset , 
                            DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
   
  SEPDriver_EndMessage(sramOffset);
            
  /* wait for the response */
  error = SEPDriver_POLL_FOR_REPONSE();
  if( error != DX_OK )
  {
    goto end_function_unlock;
  }
   
  /*-------------------
    start reading message from the SEP 
  ---------------------*/
   
  /* start the message */
  error = SEPDriver_StartIncomingMessage(&sramOffset);
  if( error != DX_OK )
  {
    goto end_function_unlock;
  }
   
  /* read opcode + status */
  error = SEPDriver_ReadParamater((DxUint32_t)messageParam,
                          sizeof(DxUint32_t) * 3,
                          sizeof(DxUint32_t) * 3, 
                          &sramOffset , 
                          DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
   
  /* check the opcode */
  if(messageParam[0] != DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_CREATE_KEY_RING_OP_CODE)
  {
    error = DX_WRONG_OPCODE_FROM_SEP_ERR;
    goto end_function_unlock;
  }
   
  /* check the status */
  if(messageParam[1] != CRYS_OK)
  {
    error = messageParam[1];
    goto end_function_unlock;
  }
  
  /* get the length of the encrypted data  */
  newKeyData_ptr->encryptedKeyDataLength = messageParam[2];
  
  /* read the encrypted key */
  maxLength = ((newKeyData_ptr->encryptedKeyDataLength + 3) / sizeof(DxUint32_t) * sizeof(DxUint32_t));
  error = SEPDriver_ReadParamater((DxUint32_t)newKeyData_ptr->encryptedKeyData_ptr,
                          newKeyData_ptr->encryptedKeyDataLength,
                          maxLength,
                          &sramOffset , 
                          DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
   
 
                                           
  /* ...................... end of function ................................ */   

end_function_unlock:   

  /* lock access to the SEP */
  SEPDriver_Unlock();

end_function:
  
  return error;

}


/*************************************************************************************************/
/*  LLF_KMNG_SendCreateUserKeyMessage ()                                                         */ 
/*************************************************************************************************/
static DxError_t LLF_KMNG_SendCreateUserKeyMessage(KMNG_KeyDataParameter_t*   keyRingData_ptr,
                                                   KMNG_KeyDataParameter_t*   newKeyData_ptr,
                                                   KMNG_KeyConfiguration_t*   keyConfiguration_ptr)
{
  /* sram offset */
  DxUint32_t  sramOffset;
  
  /* message param */
  DxUint32_t  messageParam[3];
  
  /* max length */
  DxUint32_t  maxLength;
  
  /* error */
  DxError_t   error;

  /*-------------------------------
      CODE
  ----------------------------------*/
  
  error = KMNG_OK;
  
   /* lock access to the SEP */
   error = SEPDriver_Lock();
   
   if(error != DX_OK)
   {
       goto end_function;
   }
  
  /*----------------------------
      start sending message to SEP 
  -----------------------------*/
  sramOffset = 0;
   
  /* start the message */
  SEPDriver_StartMessage(&sramOffset);
  
  /* prepare message  */
  messageParam[0] = DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_CREATE_KEY_OP_CODE;
 
  /* send message */
  error = SEPDriver_WriteParamater((DxUint32_t)messageParam,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
 
  /* send key ring data */                           
  LLF_KMNG_SendKeyRingData(keyRingData_ptr , &sramOffset);
    
  /* send new key password legth */
  error = SEPDriver_WriteParamater((DxUint32_t)&newKeyData_ptr->keyPasswordLen,
                             sizeof(DxUint32_t),
                             sizeof(DxUint32_t),
                             &sramOffset , 
                             DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                             
  /* send  new key password */
  error = SEPDriver_WriteParamater((DxUint32_t)newKeyData_ptr->keyPassword_ptr,
                             newKeyData_ptr->keyPasswordLen,
                             64*sizeof(DxUint32_t),
                             &sramOffset , 
                             DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }

  /* send  key configuration */
  error = SEPDriver_WriteParamater((DxUint32_t)keyConfiguration_ptr,
                            sizeof(DxUint32_t) * 6,
                            sizeof(DxUint32_t) * 6,
                            &sramOffset , 
                            DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
  
   
  SEPDriver_EndMessage(sramOffset);
            
  /* wait for the response */
  error = SEPDriver_POLL_FOR_REPONSE();
  if( error != DX_OK )
  {
    goto end_function_unlock;
  }
   
  /*-------------------
    start reading message from the SEP 
  ---------------------*/
   
  /* start the message */
  error = SEPDriver_StartIncomingMessage(&sramOffset);
  if( error != DX_OK )
  {
    goto end_function_unlock;
  }
   
  /* read opcode + status */
  error = SEPDriver_ReadParamater((DxUint32_t)messageParam,
                          sizeof(DxUint32_t) * 3,
                          sizeof(DxUint32_t) * 3, 
                          &sramOffset , 
                          DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
   
  /* check the opcode */
  if(messageParam[0] != DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_CREATE_KEY_OP_CODE)
  {
    error = DX_WRONG_OPCODE_FROM_SEP_ERR;
    goto end_function_unlock;
  }
   
  /* check the status */
  if(messageParam[1] != CRYS_OK)
  {
    error = messageParam[1];
    goto end_function_unlock;
  }
  
  /* get the length of the encrypted data  */
  newKeyData_ptr->encryptedKeyDataLength = messageParam[2];
  
  /* read the encrypted key */
  maxLength = ((newKeyData_ptr->encryptedKeyDataLength + 3) / sizeof(DxUint32_t) * sizeof(DxUint32_t));
  error = SEPDriver_ReadParamater((DxUint32_t)newKeyData_ptr->encryptedKeyData_ptr,
                          newKeyData_ptr->encryptedKeyDataLength,
                          maxLength, 
                          &sramOffset , 
                          DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
   
 
                                           
  /* ...................... end of function ................................ */   

end_function_unlock:   

  /* lock access to the SEP */
  SEPDriver_Unlock();

end_function:
  
  return error;

}


/*************************************************************************************************/
/*  LLF_KMNG_SendKeyRingData()                                                                   */ 
/*************************************************************************************************/

static void LLF_KMNG_SendKeyRingData(KMNG_KeyDataParameter_t*    keyRingData_ptr,
                                          DxUint32_t*                 sramOffset_ptr)
{
  /* error */
  DxError_t   error;
  
  /*------------------------
      CODE
  -------------------------*/

  /* send message */
  error = SEPDriver_WriteParamater((DxUint32_t)&keyRingData_ptr->keyPasswordLen,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           sramOffset_ptr , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      return;
  }
 
                             
  /* send  key ring password */
  error = SEPDriver_WriteParamater((DxUint32_t)keyRingData_ptr->keyPassword_ptr,
                             keyRingData_ptr->keyPasswordLen,
                             64*sizeof(DxUint32_t),
                             sramOffset_ptr , 
                             DX_FALSE);
  if(error != DX_OK)
  {
      return;
  }
                             
  /* send key ring protection key length */
  error = SEPDriver_WriteParamater((DxUint32_t)&keyRingData_ptr->encryptedKeyDataLength,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           sramOffset_ptr , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      return;
  }
                           
  /* send key ring protection key */
  error = SEPDriver_WriteParamater((DxUint32_t)keyRingData_ptr->encryptedKeyData_ptr,
                           keyRingData_ptr->encryptedKeyDataLength,
                           80*sizeof(DxUint32_t),
                           sramOffset_ptr , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      return;
  }

}

                  									 
/*************************************************************************************************/
/**
 * @brief     This function retrieves the key configuration from key ring.
 *
 *
 * @param[in]  keyRingKeyData_ptr - pointer to the the Key Ring key data
 * @param[in]  userKeyData_ptr - pointer to the data of retrieved key in the ring
 * @param[out] keyConfiguration_ptr - configuration of the retriev ed key
 * 
 * @return     DxError:  
 *                        
 */                                       
DxError_t LLF_KMNG_RetrieveKeyConfiguration(
                                    KMNG_KeyDataParameter_t*   keyRingKeyData_ptr,
                                    KMNG_KeyDataParameter_t*   userKeyData_ptr,
                                    KMNG_KeyConfiguration_t*   keyConfiguration_ptr )
{
  /* sram offset */
  DxUint32_t  sramOffset;
  
  /* message param */
  DxUint32_t  messageParam[9];
  
  /* max length */
  DxUint32_t  maxLength;
  
  /* error */
  DxError_t   error;

  /*-------------------------------
      CODE
  ----------------------------------*/

  
  error = KMNG_OK;
  
   /* lock access to the SEP */
   error = SEPDriver_Lock();
   
   if(error != DX_OK)
   {
       goto end_function;
   }
  
  /*----------------------------
      start sending message to SEP 
  -----------------------------*/
  sramOffset = 0;
   
  /* start the message */
  SEPDriver_StartMessage(&sramOffset);
  
  /* prepare message  */
  messageParam[0] = DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_RETRIEVE_KEY_CONFIGURATION_OP_CODE;
 
  /* send message */
  error = SEPDriver_WriteParamater(
                           (DxUint32_t)messageParam,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
 
  /* send key ring data */                   
  LLF_KMNG_SendKeyRingData(keyRingKeyData_ptr , &sramOffset);
                           
  /* send key data length */
  error = SEPDriver_WriteParamater(
                           (DxUint32_t)&userKeyData_ptr->encryptedKeyDataLength,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
  
                           
  /* send key data */
  maxLength = ((userKeyData_ptr->encryptedKeyDataLength + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
 
  error = SEPDriver_WriteParamater(
                           (DxUint32_t)userKeyData_ptr->encryptedKeyData_ptr,
                           userKeyData_ptr->encryptedKeyDataLength,
                           maxLength,
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                           
  SEPDriver_EndMessage(sramOffset);
            
  /* wait for the response */
  error = SEPDriver_POLL_FOR_REPONSE();
  if( error != DX_OK )
  {
    goto end_function_unlock;
  }
   
  /*-------------------
    start reading message from the SEP 
  ---------------------*/
   
  /* start the message */
  error = SEPDriver_StartIncomingMessage(&sramOffset);
  if( error != DX_OK )
  {
    goto end_function_unlock;
  }
   
  /* read opcode + status */
  error = SEPDriver_ReadParamater((DxUint32_t)messageParam,
                          sizeof(DxUint32_t) * 2,
                          sizeof(DxUint32_t) * 2, 
                          &sramOffset , 
                          DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
   
  /* check the opcode */
  if(messageParam[0] != DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_RETRIEVE_KEY_CONFIGURATION_OP_CODE)
  {
    error = DX_WRONG_OPCODE_FROM_SEP_ERR;
    goto end_function_unlock;
  }
   
  /* check the status */
  if(messageParam[1] != KMNG_OK)
  {
    error = messageParam[1];
    goto end_function_unlock;
  }
  
  /* read user specific key data */  
   error = SEPDriver_ReadParamater(
                          (DxUint32_t)keyConfiguration_ptr,
                          sizeof(KMNG_KeyConfiguration_t),
                          sizeof(KMNG_KeyConfiguration_t), 
                          &sramOffset , 
                          DX_TRUE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
 
 
end_function_unlock:

  /* unlock access to the SEP */
  SEPDriver_Unlock();

end_function:

  return error;
}


                                    
                                    
/**
 * @brief     This function updates the user specific data of the key in the key ring
 *
 *
 * @param[in]  keyRingKeyData_ptr - pointer to the the Key Ring key data
 * @param[in]  userKeyData_ptr - pointer to user key data to be updated in the ring
 * @param[out] userSpecificKeyData - new user data
 * 
 * @return     DxError:  
 *                        
 */
DxError_t LLF_KMNG_UpdateKeyUserData(KMNG_KeyDataParameter_t*      keyRingKeyData_ptr,
                                     KMNG_KeyDataParameter_t*      userKeyData_ptr,
								                     KMNG_UserSpecificKeyData_t    userSpecificKeyData )
{
  /* sram offset */
  DxUint32_t  sramOffset;
  
  /* message param */
  DxUint32_t  messageParam[9];
  
  /* max length */
  DxUint32_t  maxLength;
  
  /* error */
  DxError_t   error;

  /*-------------------------------
      CODE
  ----------------------------------*/

  
  error = KMNG_OK;
  
   /* lock access to the SEP */
   error = SEPDriver_Lock();
   
   if(error != DX_OK)
   {
       goto end_function;
   }
  
  /*----------------------------
      start sending message to SEP 
  -----------------------------*/
  sramOffset = 0;
   
  /* start the message */
  SEPDriver_StartMessage(&sramOffset);
  
  /* prepare message  */
  messageParam[0] = DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_UPDATE_KEY_SPECIF_DATA_OP_CODE;
 
  /* send message */
  error = SEPDriver_WriteParamater((DxUint32_t)messageParam,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
 
  /* send key ring data */                   
  LLF_KMNG_SendKeyRingData(keyRingKeyData_ptr , &sramOffset);
  
  /* key password length */ 
  error = SEPDriver_WriteParamater((DxUint32_t)&userKeyData_ptr->keyPasswordLen,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
  
  /* send key password */                         
  error = SEPDriver_WriteParamater(
                           (DxUint32_t)userKeyData_ptr->keyPassword_ptr,
                           userKeyData_ptr->keyPasswordLen,
                           64*sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                           
  /* send key data length */
  error = SEPDriver_WriteParamater(
                           (DxUint32_t)&userKeyData_ptr->encryptedKeyDataLength,
                           sizeof(DxUint32_t),
                           sizeof(DxUint32_t),
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }  
                           
  /* send key data */
  maxLength = ((userKeyData_ptr->encryptedKeyDataLength + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
 
  error = SEPDriver_WriteParamater(
                           (DxUint32_t)userKeyData_ptr->encryptedKeyData_ptr,
                           userKeyData_ptr->encryptedKeyDataLength,
                           maxLength,
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
 
  /* send updated user specific data */
  error = SEPDriver_WriteParamater(
                           (DxUint32_t)userSpecificKeyData,
                           KMNG_USER_SPECIFIC_KEY_DATA_SIZE,
                           KMNG_USER_SPECIFIC_KEY_DATA_SIZE,
                           &sramOffset , 
                           DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
 
                           
  SEPDriver_EndMessage(sramOffset);
            
  /* wait for the response */
  error = SEPDriver_POLL_FOR_REPONSE();
  if( error != DX_OK )
  {
    goto end_function_unlock;
  }
   
  /*-------------------
    start reading message from the SEP 
  ---------------------*/
   
  /* start the message */
  error = SEPDriver_StartIncomingMessage(&sramOffset);
  if( error != DX_OK )
  {
    goto end_function_unlock;
  }
   
  /* read opcode + status */
  error = SEPDriver_ReadParamater((DxUint32_t)messageParam,
                          sizeof(DxUint32_t) * 2,
                          sizeof(DxUint32_t) * 2, 
                          &sramOffset , 
                          DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
   
  /* check the opcode */
  if(messageParam[0] != DX_SEP_HOST_SEP_PROTOCOL_HOST_KMNG_UPDATE_KEY_SPECIF_DATA_OP_CODE)
  {
    error = DX_WRONG_OPCODE_FROM_SEP_ERR;
    goto end_function_unlock;
  }
   
  /* check the status */
  if(messageParam[1] != KMNG_OK)
  {
    error = messageParam[1];
    goto end_function_unlock;
  }
  
  /* read the size of the data */
  error = SEPDriver_ReadParamater((DxUint32_t)&userKeyData_ptr->encryptedKeyDataLength,
                                  sizeof(DxUint32_t),
                                  sizeof(DxUint32_t),
                                  &sramOffset,
                                  DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
                          
  /* read the encrypted message itself */
  maxLength = ((userKeyData_ptr->encryptedKeyDataLength + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
  error = SEPDriver_ReadParamater((DxUint32_t)userKeyData_ptr->encryptedKeyData_ptr,
                                   userKeyData_ptr->encryptedKeyDataLength,
                                   maxLength,
                                   &sramOffset , 
                                   DX_FALSE);
  if(error != DX_OK)
  {
      goto end_function_unlock;
  }
 
end_function_unlock:

  /* unlock access to the SEP */
  SEPDriver_Unlock();

end_function:

  return error;

}
